-- Copyright (c) 2001, 2009, Oracle and/or its affiliates. 
-- All rights reserved. 

---------------------------------------------------------------------------
--  Package name:
--      photo_album
--
--  Description:
--      The photo_album PL/SQL package implements a simple photo album 
--      sample application using the Oracle Multimedia ORDImage type and the 
--      PL/SQL Web Toolkit.
---------------------------------------------------------------------------

---------------------------------------------------------------------------
--  Package specification.
---------------------------------------------------------------------------
CREATE OR REPLACE PACKAGE photo_album
AS
    --
    -- The public interface for the photo_album application
    --
    -- the view_ procedures are web navigation entry points
    PROCEDURE view_album( q IN VARCHAR2 DEFAULT NULL );
    PROCEDURE view_entry( entry_id IN VARCHAR2 );
    PROCEDURE view_metadata( entry_id IN VARCHAR2 );
    PROCEDURE view_upload_form;
    PROCEDURE view_metadata_form( entry_id IN VARCHAR2 );
    PROCEDURE view_search_form;

    -- These perform work, accepting new input or delivering content
    -- from the database
    PROCEDURE deliver_media( entry_id IN VARCHAR2, media IN VARCHAR2 );
    PROCEDURE insert_new_photo( new_description IN VARCHAR2, 
                                new_photo IN VARCHAR2 );
    PROCEDURE write_metadata( entry_id IN VARCHAR2, 
                              title    IN VARCHAR2,
                              creator  IN VARCHAR2,
                              date     IN VARCHAR2,
                              description  IN VARCHAR2,
                              copyright IN VARCHAR2 );
    PROCEDURE search_metadata( mtype    IN VARCHAR2,
                               tag      IN VARCHAR2,
                               op       IN VARCHAR2 DEFAULT 'contains',
                               search   IN VARCHAR2 );

    -- this function is used to return CLOB types to create a multi-column
    -- datastore for full text indexing.
    FUNCTION getClob( xmlDoc IN XMLType ) RETURN CLOB;

    -- Specify this function for the PlsqlRequestValidationFunction
    -- parameter in mod_plsql DAD configuration. Restricts the PhotoAlbum DAD
    -- to execute only the photo_album package;
    FUNCTION validate_request( procedure_name IN VARCHAR2 ) RETURN BOOLEAN;

END photo_album;
/
SHOW ERRORS;


---------------------------------------------------------------------------
--  Package body specification.
---------------------------------------------------------------------------
CREATE OR REPLACE PACKAGE BODY photo_album AS

    --
    -- define record and cursor type used to view the photo album
    --
    TYPE entry_type IS RECORD 
    (
      id          photos.id%TYPE,
      description photos.description%TYPE,
      thumb       photos.thumb%TYPE
    );
    TYPE album_cur_type IS REF CURSOR RETURN entry_type;

    --
    -- Forward declarations for private methods
    --
    FUNCTION  get_preferred_format( format IN VARCHAR2 ) RETURN VARCHAR2;
    PROCEDURE sync_indexes;
    PROCEDURE print_album( album_cur IN album_cur_type,
                           empty_msg IN VARCHAR2 );
    PROCEDURE print_search_form( mtype  IN VARCHAR2 DEFAULT 'exif',
                                 tag    IN VARCHAR2 DEFAULT NULL,
                                 op     IN VARCHAR2 DEFAULT 'contains',
                                 search IN VARCHAR2 DEFAULT NULL );
    PROCEDURE print_upload_form;
    PROCEDURE print_metadata_form( entry_id IN VARCHAR2 );
    PROCEDURE print_page_header( heading_tag IN VARCHAR2 DEFAULT NULL );
    PROCEDURE print_image_link( type     IN VARCHAR2, 
                                entry_id IN VARCHAR2,
                                alt      IN VARCHAR2 DEFAULT NULL, 
                                height   IN INTEGER  DEFAULT 0,
                                width    IN INTEGER  DEFAULT 0,
                                attrs    IN VARCHAR2 DEFAULT NULL );
    PROCEDURE print_nav_links( nav_tag IN VARCHAR2, 
                               entry_id IN VARCHAR2 DEFAULT NULL,
                               search IN VARCHAR2 DEFAULT NULL );
    PROCEDURE print_page_trailer( display_return_url IN BOOLEAN DEFAULT FALSE );
    PROCEDURE print_heading( message IN VARCHAR2 );
    PROCEDURE print_error( message IN VARCHAR2 );
    PROCEDURE print_link( link IN VARCHAR2, message IN VARCHAR2 );
    PROCEDURE print_metadata ( meta IN XMLType );

    -- some useful constants
    nbsp VARCHAR2( 6 )      := CHR( 38 ) || 'nbsp;';
    ampersand VARCHAR2( 1 ) := CHR( 38 );

---------------------------------------------------------------------------
--  Procedure name:
--      view_album
--
--  Description:
--      Display the contents of the photo album with thumbnail images for 
--      all the entries that match the keyword search criteria. If search is 
--      NULL then return all entries.
---------------------------------------------------------------------------
PROCEDURE view_album( q IN VARCHAR2 DEFAULT NULL )
IS
    album_cur    album_cur_type;
    where_clause VARCHAR2(128);
    search       VARCHAR2(40) := trim(substr( q, 1, 40 ));
    nav_tag      VARCHAR2(12);
BEGIN
    --
    -- Output common page header
    -- and navigation links
    print_page_header;

    IF search IS NULL THEN
      print_nav_links( 'v_album' );
    ELSE
      print_nav_links( nav_tag=>'s_album', search=>search );
    END IF;

    --
    -- no search criteria so fetch all entries
    --
    IF search IS NULL THEN 
      OPEN album_cur FOR 
        SELECT id, description, thumb 
        FROM photos 
        ORDER BY id;
      print_album( album_cur, 'The photo album is empty.' );
      CLOSE album_cur;
    ELSE
    -- 
    -- use the full-text index to select entries matching the search criteria
    --

      OPEN album_cur FOR 
        SELECT id, description, thumb 
        FROM photos 
        WHERE CONTAINS( description, trim(search) ) > 0
        ORDER BY id;

      print_album( album_cur, 'No photos were found.' );
      CLOSE album_cur;
    END IF;

    --
    -- Output common page trailer
    --
    print_page_trailer;
END view_album;


---------------------------------------------------------------------------
--  Procedure name:
--      view_entry
--
--  Description:
--      View the full-size version of an image.
---------------------------------------------------------------------------
PROCEDURE view_entry( entry_id IN VARCHAR2 )
IS
    sc_description VARCHAR2(256);
    photo          ORDImage;
    metaURL        VARCHAR2(4000);
BEGIN

    --
    -- Output common page header 
    --
    print_page_header;

    --
    -- Fetch the row.
    --
    BEGIN
      SELECT htf.escape_sc(description), image
      INTO sc_description, photo
      FROM photos
      WHERE id = entry_id;
      EXCEPTION
      WHEN no_data_found THEN
        print_error( 'Image <b>' || htf.escape_sc(entry_id) || 
                    '</b> was not found.</p>' );
        print_page_trailer( TRUE );
        return;
    END;

    --
    -- print navigation links
    --
    print_nav_links( 'entry', entry_id );

    --
    -- Display the data in a table
    --
    htp.print( '<p><table summary="View image entry">' );
    htp.print( '<tr><td scope="col"><b>Description:</b></td>' );
    htp.print( '<td scope="col">' || sc_description || '</td></tr>' );

    htp.print( '<tr><td scope="col" valign="top"><b>Photo:</b></td>' );
    htp.prn( '<td scope="col">' );
    print_image_link( 'image', entry_id, sc_description, 
                      photo.height, photo.width );

    htp.print( '</td></tr></table></p>' );

    --
    -- Output common page trailer
    --
    print_page_trailer( FALSE );

END view_entry;

---------------------------------------------------------------------------
--  Procedure name:
--      view_meta
--
--  Description:
--      View the metadata of an image.
---------------------------------------------------------------------------
PROCEDURE view_metadata( entry_id IN VARCHAR2 )
IS
  metaO XMLType;
  metaE XMLType;
  metaI XMLType;
  metaX XMLType;
  meta  XMLType;
  metaT VARCHAR2(10);
  noMeta BOOLEAN := FALSE;
BEGIN

    --
    -- Output common page header.
    -- 
    print_page_header;

    --
    -- Fetch the row.
    --
    BEGIN
      SELECT metaOrdImage, metaEXIF, metaIPTC, metaXMP
      INTO   metaO, metaE, metaI, metaX
      FROM   photos
      WHERE  id = entry_id;

      EXCEPTION
      WHEN no_data_found THEN
        print_error( 'Image <b>' || htf.escape_sc(entry_id) || 
                    '</b> was not found.</p>' );
        print_page_trailer( TRUE );
        return;
    END;

    -- is there any metadata ?
    noMeta := metaO IS NULL AND metaI IS NULL AND
              metaE IS NULL AND metaX IS NULL;

    --
    -- print navigation links
    --
    print_nav_links( 'view_metadata', entry_id );
    htp.print( '<div>' );
    print_image_link( type=>'thumb', 
                      entry_id=>entry_id, 
                      alt=>'thumbnail of image ' || entry_id,
                      attrs=>'align="left"' );

    IF noMeta = TRUE THEN
      htp.print( nbsp || '<i>No metadata is available.</i>' );
    ELSE
      htp.print( nbsp || '<i>The following metadata is available:</i><br>' ||
                 nbsp );

      IF metaE IS NOT NULL THEN 
        htp.anchor( curl=>'#exifMetadata', ctext=>'EXIF' );
      END IF;
      IF metaI IS NOT NULL THEN 
        htp.anchor( curl=>'#iptcMetadata', ctext=>'IPTC' );
      END IF;
      IF metaX IS NOT NULL THEN 
        htp.anchor( curl=>'#xmpMetadata', ctext=>'XMP' );
      END IF;
      IF metaO IS NOT NULL THEN 
        htp.anchor( curl=>'#ordImageAttributes', ctext=>'ORDIMAGE' );
      END IF;
      htp.print( '</div>' );
    END IF;
    htp.print('<br clear="left"><p>' );

    -- display the metadata
    IF metaE IS NOT NULL THEN 
      htp.print( '<span class="bigBlue" id="exifMetadata">EXIF</span>' );
      htp.print( '<br><pre>' );
      print_metadata( metaE ); 
      htp.print( '</pre>' );
    END IF;
    IF metaI IS NOT NULL THEN 
      htp.print( '<span class="bigBlue" id="iptcMetadata">IPTC</span>' );
      htp.print( '<br><pre>' );
      print_metadata( metaI ); 
      htp.print( '</pre>' );
    END IF;
    IF metaX IS NOT NULL THEN 
      htp.print( '<span class="bigBlue" id="xmpMetadata">XMP</span>' );
      htp.print( '<br><pre>' );
      print_metadata( metaX ); 
      htp.print( '</pre>' );
    END IF;
    IF metaO IS NOT NULL THEN 
      htp.print( '<span class="bigBlue" id="ordImageAttributes">
                   ORDIMAGE</span>' );
      htp.print( '<br><pre>' );
      print_metadata( metaO ); 
      htp.print( '</pre>' );
    END IF;

    --
    -- Output common page trailer
    --
    print_page_trailer;

END view_metadata;

---------------------------------------------------------------------------
--  Procedure name:
--      view_upload_form
--
--  Description:
--      Display the upload form.
---------------------------------------------------------------------------
PROCEDURE view_upload_form
IS
BEGIN
    --
    -- Output the form.
    --
    print_page_header;
    print_nav_links( 'upload' );
    print_upload_form;
    print_page_trailer;
END view_upload_form;

---------------------------------------------------------------------------
--  Procedure name:
--      view_metadata_form
--
--  Description:
--      Display the metadata_form
---------------------------------------------------------------------------
PROCEDURE view_metadata_form( entry_id IN VARCHAR2 )
IS
BEGIN
    --
    -- Output the form.
    --
    print_page_header;
    print_nav_links( 'write_metadata', entry_id );
    print_metadata_form( entry_id );
    print_page_trailer;
END view_metadata_form;

---------------------------------------------------------------------------
--  Procedure name:
--      view_search_form
--
--  Description:
--      Display the search_form
---------------------------------------------------------------------------
PROCEDURE view_search_form
IS
BEGIN
    --
    -- Output the form.
    --
    print_page_header;
    print_nav_links( 'search_metadata' );
    print_search_form;
    print_page_trailer;
END view_search_form;

---------------------------------------------------------------------------
--  Procedure name:
--      deliver_media
--
--  Description:
--      Deliver a thumbnail or full-size image to the browser.
---------------------------------------------------------------------------
PROCEDURE deliver_media( entry_id IN VARCHAR2, media IN VARCHAR2 ) 
IS
    local_image ORDSYS.ORDIMAGE;
BEGIN
    --
    -- Fetch the thumbnail or full-size image from the database.
    --
    IF media = 'thumb'
    THEN
        SELECT thumb INTO local_image FROM photos WHERE id = entry_id;
    ELSE
        SELECT image INTO local_image FROM photos WHERE id = entry_id;
    END IF;

    --
    -- Check update time if browser sent If-Modified-Since header
    --
    IF ordplsgwyutil.cache_is_valid( local_image.getUpdateTime() )
    THEN
      owa_util.status_line( ordplsgwyutil.http_status_not_modified );
      RETURN;
    END IF;

    --
    -- Set the MIME type and deliver the image to the browser.
    --
    owa_util.mime_header( local_image.mimeType, FALSE );
    ordplsgwyutil.set_last_modified( local_image.getUpdateTime() );
    owa_util.http_header_close();

    IF owa_util.get_cgi_env( 'REQUEST_METHOD' ) <> 'HEAD' THEN  
      wpg_docload.download_file( local_image.source.localData );
    END IF;

END deliver_media;



---------------------------------------------------------------------------
--  Procedure name:
--      write_metadata
--
--  Description:
--      Write XMP metadata to a photo
---------------------------------------------------------------------------
PROCEDURE write_metadata( entry_id    IN VARCHAR2,
                          title       IN VARCHAR2,
                          creator     IN VARCHAR2,
                          date        IN VARCHAR2,
                          description IN VARCHAR2,
                          copyright   IN VARCHAR2 )
IS
  img ORDSYS.ORDImage := NULL;
  xmp XMLType;
  buf VARCHAR2(5000);
  doc dbms_xmldom.DOMDocument;
  elem dbms_xmldom.DOMElement;
  node dbms_xmldom.DOMNode;
  des  photos.description%TYPE;

  -- work with cleaned up input
  c_title       VARCHAR2(40) := trim(substr(title,       1, 40));
  c_creator     VARCHAR2(40) := trim(substr(creator,     1, 40));
  c_date        VARCHAR2(40) := trim(substr(date,        1, 40));
  c_description VARCHAR2(40) := trim(substr(description, 1, 40));
  c_copyright   VARCHAR2(40) := trim(substr(copyright,   1, 40));

BEGIN
    --
    -- Make sure a title has been provided. If not, display an error 
    -- message, then redisplay the form.
    --
    IF c_title IS NULL 
    THEN
        print_page_header;
        print_error( 'Please supply a title.</p>' );
        print_metadata_form( entry_id );
        print_page_trailer( TRUE );
        return;
    END IF;

    -- Create the XMP packet it must be schema valid
    -- to "http://xmlns.oracle.com/ord/meta/xmp"
    -- and contain an <RDF> element. This example uses
    -- the Dublin Core schema as implemented by Adobe XMP
    buf := '<xmpMetadata xmlns="http://xmlns.oracle.com/ord/meta/xmp" 
             xsi:schemaLocation="http://xmlns.oracle.com/ord/meta/xmp 
             http://xmlns.oracle.com/ord/meta/xmp" 
             xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" >
      <rdf:RDF xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#">
      <rdf:Description about="" xmlns:dc="http://purl.org/dc/elements/1.1/">
      <dc:title>' || htf.escape_sc(title) || '</dc:title>';

    IF c_creator IS NOT NULL THEN
      buf := buf || '<dc:creator>' || htf.escape_sc(c_creator) 
                 || '</dc:creator>';
    END IF;
    IF c_date IS NOT NULL THEN
      buf := buf || '<dc:date>' || htf.escape_sc(c_date) 
                 || '</dc:date>';
    END IF;
    IF c_description IS NOT NULL THEN
      buf := buf || '<dc:description>' || htf.escape_sc(c_description) 
                 || '</dc:description>';
    END IF;
    IF c_copyright IS NOT NULL THEN
      buf := buf || '<dc:copyright>' || htf.escape_sc(c_copyright) 
                 || '</dc:copyright>';
    END IF;
    buf := buf || '
      </rdf:Description>
      </rdf:RDF>
      </xmpMetadata>';

    xmp := XMLType.createXML(buf, 'http://xmlns.oracle.com/ord/meta/xmp');

    -- 
    -- Select image for update.
    -- The description is selected to force update of CTX index.
    --
    SELECT image, description
    INTO img, des
    FROM photos 
    WHERE id = entry_id 
    FOR UPDATE;

    --
    -- write the metadata
    --
    img.putMetadata( xmp, 'XMP' );

    --
    -- Save updated image and new metadata to table.
    -- The description is updated to force update of CTX index.
    --
    UPDATE photos
    SET image = img,
        metaXMP = xmp,
        description = des
    WHERE id = entry_id;

    -- Update the text indexes.
    sync_indexes;
    COMMIT;
    
    --
    -- Redirect browser to display full album.
    -- 
    print_page_header( 
        '<meta http-equiv="refresh" content="2;url=PHOTO_ALBUM.VIEW_ALBUM">' );
    print_heading( 'Metadata successfully written' );
    htp.print( '<hr size="-1"> <p>Please click on the link below or ' ||
               'wait for the browser to refresh the page.</p>' );
    print_page_trailer( TRUE );
    
END write_metadata;

---------------------------------------------------------------------------
--  Procedure name:
--      search_metadata
--
--  Description:
--      Perform a schema based search
---------------------------------------------------------------------------
PROCEDURE search_metadata( mtype    IN VARCHAR2,
                           tag      IN VARCHAR2,
                           op       IN VARCHAR2 DEFAULT 'contains',
                           search   IN VARCHAR2 )
IS
  album_cur album_cur_type;
  errorMsg VARCHAR2(64) := NULL;
  xpath    VARCHAR2(400);
  nspace   VARCHAR2(200);
  xquery   VARCHAR2(1000);
  c_search  VARCHAR2(40) := trim(substr(search, 1, 40));

BEGIN
    print_page_header;
    print_nav_links( 'search_metadata' );

    -- Set up search variables for EXIF documents.
    IF mtype = 'exif' THEN
      IF op = 'equals' THEN
        xpath  := '/exifMetadata[//' || tag || '="' || c_search || '"]';
      ELSE  -- default to contains
        xpath  := '/exifMetadata//' || tag || 
                  '[contains(., "' || c_search || '")]';
      END IF;

      xquery := 'declare default element namespace ' ||
                ' "http://xmlns.oracle.com/ord/meta/exif"; $x' || xpath;

      OPEN album_cur FOR
        SELECT id, description, thumb
        FROM photos 
        WHERE xmlexists(xquery passing metaExif as "x");

    -- Set up search variables for IPTC documents.
    ELSIF mtype = 'iptc' THEN
      IF op = 'equals' THEN
        xpath  := '/iptcMetadata[//' || tag || '="' || c_search || '"]';
      ELSE  -- default to contains
        xpath  := '/iptcMetadata//' || tag || 
                  '[contains(., "' || c_search || '")]';
      END IF;

      xquery := 'declare default element namespace ' ||
            ' "http://xmlns.oracle.com/ord/meta/iptc"; $x' || xpath;

      OPEN album_cur FOR
        SELECT id, description, thumb
        FROM photos 
        WHERE xmlexists(xquery passing metaIptc as "x");

    -- Set up search variables for XMP documents.
    ELSIF mtype = 'xmp' THEN
      -- default to contains
      xpath  := '//rdf:Description//*[contains(., "' 
                || c_search || '")]';

      -- Add rdf namespace prefix.
      xquery := 'declare namespace rdf = ' || 
             ' "http://www.w3.org/1999/02/22-rdf-syntax-ns#"; ' ||
             'declare default element namespace ' ||
             ' "http://xmlns.oracle.com/ord/meta/xmp"; $x' || xpath;

      OPEN album_cur FOR
        SELECT id, description, thumb
        FROM photos 
        WHERE xmlexists(xquery passing metaXMP as "x");

    ELSE
      errorMsg := 'Search domain is invalid: ' || htf.escape_sc(mtype);
    END IF;

    print_search_form( mtype, tag, op, c_search );
    htp.print('<hr size="-1">');
    print_album( album_cur, 'No photos matched the search criteria.' );    
    CLOSE album_cur;

END search_metadata;


---------------------------------------------------------------------------
--  Procedure name:
--      insert_new_photo
--
--  Description:
--      Insert a new photo into the album.
---------------------------------------------------------------------------
PROCEDURE insert_new_photo( new_description IN VARCHAR2, 
                            new_photo       IN VARCHAR2 )
IS
    c_description VARCHAR2( 40 ) := trim(substr( new_description, 1, 40 ));
    upload_size INTEGER;
    upload_mime_type VARCHAR2( 128 );
    upload_blob BLOB;
    new_id NUMBER;
    new_image ORDSYS.ORDIMAGE;
    new_thumb ORDSYS.ORDIMAGE;
    pos INTEGER;
    image_blob BLOB;
    thumb_scale varchar2(32) := 'maxScale=75,75';
    meta_root VARCHAR2(100);
    xmlORD  XMLType;
    xmlIPTC XMLType;
    xmlEXIF XMLType;
    xmlXMP  XMLType;
    metav   XMLSequenceType;
BEGIN
    --
    -- Make sure a file name has been provided. If not, display an error 
    -- message, then redisplay the form.
    --
    IF new_photo IS NULL OR LENGTH( new_photo ) = 0
    THEN
        print_page_header;
        print_error( 'Please supply a file name.' );
        print_upload_form;
        print_page_trailer( TRUE );
        return;
    END IF;

    --
    -- Get the length, MIME type and the BLOB of the new photo from the 
    -- upload table.
    --
    SELECT doc_size, 
           mime_type, 
           blob_content 
    INTO   upload_size, 
           upload_mime_type, 
           upload_blob
    FROM photos_upload 
    WHERE name = new_photo;

    --
    -- Make sure we have a valid file.
    --
    IF upload_size = 0
    THEN
        print_page_header;
        print_heading( 'Error message' );
        htp.print( '<hr size="-1"><p>Please supply a valid image file.</p>' );
        print_upload_form;
        print_page_trailer( TRUE );
        return;
    END IF;

    --
    -- If the description is blank, then use the file name.
    --
    IF c_description IS NULL 
    THEN
        c_description := new_photo;
        pos := INSTR( c_description, '/', -1 );
        IF pos > 0
        THEN
            c_description := SUBSTR( c_description, pos + 1 );
        END IF;
        c_description := SUBSTR( 'Image from file: ' ||   
                         c_description || '.', 1, 40 );
    END IF;

    --
    -- Insert a new row into the table, returning the newly allocated sequence 
    -- number. 
    INSERT INTO photos ( id, description, metaExif, metaIPTC, metaXMP, 
                         image, thumb )
    VALUES ( photos_sequence.nextval, c_description, NULL, NULL, NULL,
                 ORDSYS.ORDIMAGE.INIT(), ORDSYS.ORDIMAGE.INIT() ) 
    RETURN id 
    INTO new_id;

    --
    -- Fetch the newly initialized full-size and thumbnail image objects.
    --
    SELECT image, 
           thumb 
    INTO new_image, 
         new_thumb 
    FROM photos 
    WHERE id = new_id 
    FOR UPDATE;

    --
    -- Load the photo from the upload table into the image object.
    --
    DBMS_LOB.COPY( new_image.source.localData, upload_blob, upload_size );
    new_image.setLocal();

    --
    -- Set the properties. If the image format is not recognized, then
    -- the exception handler will set the MIME type and length from the
    -- upload table.
    --
    BEGIN
        new_image.setProperties();
    EXCEPTION
        WHEN OTHERS THEN
             new_image.contentLength := upload_size;
             new_image.mimeType := upload_mime_type;
    END;

    --
    -- Some image formats are supported by Oracle Multimedia but can not
    -- be displayed inline by a browser. The BMP format is one example.
    -- Convert the image to a GIF or JPEG based on number of colors in the
    -- image.
    --
    IF new_image.contentFormat IS NOT NULL AND
       ( new_image.mimeType = 'image/bmp' OR
         new_image.mimeType = 'image/x-bmp' )
    THEN
        BEGIN
            new_image.process( 
                         'fileFormat=' ||
                         get_preferred_format( new_image.contentFormat ) );
        EXCEPTION
            WHEN OTHERS THEN
                NULL;
        END;
    END IF;

    --
    -- Try to copy the full-size image and process it to create the thumbnail.
    -- This may not be possible if the image format is not recognized.
    --
    BEGIN
        new_image.processCopy( thumb_scale, new_thumb );
    EXCEPTION
        WHEN OTHERS THEN
            new_thumb.deleteContent();
            new_thumb.contentLength := 0;
    END;

    --
    -- Fetch the metadata and sort the results.
    --
    BEGIN
      metav := new_image.getMetadata( 'ALL' );
      FOR i IN 1..metav.count() LOOP
        select xmlcast(xmlquery('fn:local-name($x)' 
          passing metav(i) as "x" returning content) as varchar2(100))
        into meta_root from dual;
        CASE meta_root
          WHEN 'ordImageAttributes' THEN xmlORD := metav(i);
          WHEN 'xmpMetadata'  THEN xmlXMP  := metav(i);
          WHEN 'iptcMetadata' THEN xmlIPTC := metav(i);
          WHEN 'exifMetadata' THEN xmlEXIF := metav(i);
          ELSE NULL;  
        END CASE;
      END LOOP;
    EXCEPTION
      WHEN OTHERS THEN    
        NULL;
    END;

    --
    -- Update the full-size and thumbnail images in the database.
    -- Update metadata columns.
    --
    UPDATE photos 
    SET image = new_image, 
        thumb = new_thumb,
        metaORDImage = xmlORD,
        metaEXIF = xmlEXIF,
        metaIPTC = xmlIPTC,
        metaXMP = xmlXMP
    WHERE id = new_id;

    -- 
    -- Update the text indexes.
    -- 
    sync_indexes;
    
    --
    -- Delete the row from the upload table.
    --
    DELETE FROM photos_upload WHERE name = new_photo;
    COMMIT;

    --
    -- Redirect browser to display full album.
    -- 
    print_page_header( 
        '<meta http-equiv="refresh" content="2;url=PHOTO_ALBUM.VIEW_ALBUM">' );
    print_heading( 'Photo successfully uploaded into photo album' );
    htp.print( '<hr size="-1"><p>Please click on the link below or ' ||
               'wait for the browser to refresh the page.</p>' );
    print_page_trailer( TRUE );

END insert_new_photo;


---------------------------------------------------------------------------
--  Procedure name:
--      sync_indexes
--
--  Description:
--      Make calls to ctx_ddl.sync_index to force updates to text index.
--
---------------------------------------------------------------------------
PROCEDURE sync_indexes
IS
BEGIN
    ctx_ddl.sync_index( 'pa_ctx_idx');
END sync_indexes;


---------------------------------------------------------------------------
--  Procedure name:
--      get_preferred_format
--
--  Description:
--      Return the preferred file format to which image formats not 
--      supported by a browser should be converted.
---------------------------------------------------------------------------
FUNCTION get_preferred_format( format IN VARCHAR2 ) RETURN VARCHAR2
IS
    num_digits INTEGER;
    ch CHAR(1);
BEGIN
    --
    -- Image content format strings have the following format:
    --   <#bits><format>
    --   MONOCHROME
    -- Figure out the number of digits that represent the number of colors.
    --
    num_digits := 0;
    LOOP
        ch := SUBSTR( format, num_digits + 1, 1 );
        IF ch >= '0' AND ch <= '9'
        THEN
            num_digits := num_digits + 1;
        ELSE
            EXIT;
        END IF;
    END LOOP;

    --
    -- Images with more than 8 bits of color can be converted to the JPEG
    -- format without significant discernible loss of quality.
    --
    IF num_digits > 0
    THEN
        IF TO_NUMBER( SUBSTR( format, 1, num_digits ) ) > 8
        THEN
            RETURN 'JFIF';
        END IF;
    END IF;

    --
    -- Images with 8 bits of color or less are best converted to the GIF
    -- format to retain the quality.
    --
    RETURN 'GIFF';

END get_preferred_format;


---------------------------------------------------------------------------
--  Procedure name:
--      print_album
--
--  Description:
--      Prints a grid of photo thumbnails with links to metadata and
--      the full-size image.
---------------------------------------------------------------------------
PROCEDURE print_album( album_cur album_cur_type, empty_msg IN VARCHAR2 )
IS
    entry       entry_type;
    entry_count INTEGER := 0;
    grid_size   INTEGER := 4;
    colIdx      INTEGER := 0;
    url         VARCHAR2(128);
    sc_description VARCHAR2(256);
BEGIN
    --
    -- Display the thumbnail images in a grid
    --
    htp.print( '<p><table border="0" cellpadding="4 cellspacing="4" ' ||
               'width="100%" summary="Table of thumbnail images">' );
    htp.tableHeader( cattributes=>'id="c0"' );
    htp.tableHeader( cattributes=>'id="c1"' );
    htp.tableHeader( cattributes=>'id="c2"' );
    htp.tableHeader( cattributes=>'id="c3"' );
    --
    -- For each entry...
    --
  BEGIN

    LOOP
        FETCH album_cur INTO entry;
        EXIT WHEN album_cur%NOTFOUND;

        --
        -- Count album entries
        --
        entry_count := entry_count + 1;

        IF entry_count = 1 THEN
          -- instructions
          htp.print( '<tr><td headers="c0 c1 c2 c3" align="center" colspan="' 
                          || grid_size || '">' );
          htp.print( '<i>Select the thumbnail to view the full-size image.' ||
               ' Select the description link to view image metadata.</i>' ); 
          htp.print( '</td></tr>' );
        END IF;

        --
        -- Start the table row 
        --
        colIdx := MOD( entry_count-1, grid_size );
        IF 0 = colIdx
        THEN
          htp.prn( '<tr>' );
        END IF;

        -- Escape the description text.
        sc_description := htf.escape_sc( entry.description );

        --
        -- Display the thumbnail image as an anchor tag which can be used
        -- to display the full-size image. If the image format isn't
        -- supported by Oracle Multimedia, then a thumbnail wouldn't have been
        -- produced when the image was uploaded, so use the text '[view
        -- image]' instead of the thumbnail.
        --

        htp.print( '<td headers="c' || colIdx || '" align="center" >
                    <a href="PHOTO_ALBUM.VIEW_ENTRY?entry_id=' || 
                    entry.id || '">' );
        IF entry.thumb.contentLength > 0 
        THEN
            print_image_link( 'thumb', entry.id, sc_description, 
                              entry.thumb.height, entry.thumb.width );
        ELSE
            htp.prn( '[view image]' );
        END IF;
        htp.print( '</a>' );

        -- Create link to the metadata.
        htp.prn('<br>');
        htp.anchor( curl=>'PHOTO_ALBUM.VIEW_METADATA?entry_id=' || entry.id,
                    ctext=>sc_description );
        htp.prn('</td>');  

        IF 3 = colIdx
        THEN
          htp.print( '</tr>' );
        END IF;

    END LOOP;
    EXCEPTION
    WHEN OTHERS THEN
       NULL;
  END;
    --
    -- Display a message if the album is empty or
    -- no entries matched search string
    --
    IF entry_count = 0 
    THEN
        htp.print( '<tr><td class="blue" scope="col" colspan="' || grid_size 
               || '" align="center">' );
        htp.print( '<b><i>' || empty_msg || '</i></b>' );
        htp.print( '</font></td></tr>' );
    ELSE
      --
      -- Finish a partial row if it's not the first.
      --
      FOR i in colIdx+1..grid_size-1 LOOP
        htp.tabledata( cvalue=>nbsp, 
                       cattributes=>'width="25%" headers="c' || i || '"' );
      END LOOP;
      htp.print( '</tr>' );
    END IF;

    --
    -- Finish the table.
    --
    htp.print( '</table></p>' );
END print_album;

---------------------------------------------------------------------------
--  Procedure name:
--      print_script
--
--  Description:
--      helper function for print_search_form
---------------------------------------------------------------------------
PROCEDURE print_script  
IS
  theScript VARCHAR2(32767);
BEGIN
  theScript := '
  <noscript>
    "A Javascript enabled browser is required for this page 
    to function properly."
  </noscript>
  <script type="text/javascript">
  <!--
   var meta_types = [ "EXIF|exif", "IPTC|iptc", "XMP|xmp" ];

   var exif_tags = [ "ApertureValue", "Artist", "BitsPerSample",
    "BrightnessValue", "ColorSpace", "ComponentsConfiguration",
    "CompressedBitsPerPixel", "Compression", "Contrast",
    "Copyright", "CustomRendered", "DateTime", "DateTimeDigitized",
    "DateTimeOriginal", "DigitalZoomRatio", "ExifVersion",
    "ExposureBiasValue", "ExposureIndex", "ExposureMode",
    "ExposureProgram", "ExposureTime", "FNumber", "FileSource",
    "Flash/Fired", "Flash/Return", "Flash/Mode", "Flash/Function",
    "Flash/RedEyeReduction",
    "FlashEnergy", "FlashPixVersion", "FocalLength",
    "FocalLengthIn35mmFilm", "FocalPlaneResolutionUnit", 
    "FocalPlaneXResolution", "FocalPlaneYResolution", "GainControl", 
    "ImageDescription", "ImageLength", "ImageWidth", "LightSource", 
    "Make", "MaxApertureValue", "MeteringMode",
    "Model", "Orientation", "PhotometricInterpretation", "PixelXDimension",
    "PixelYDimension", "PlanarConfiguration", "RelatedSoundFile",
    "ResolutionUnit", "SamplesPerPixel", "Saturation",
    "SceneCaptureType", "SceneType", "SensingMethod",
    "Sharpness", "ShutterSpeedValue",  "Software",
    "SpectralSensitivity",  "SubjectArea",
    "SubjectDistance",  "SubjectDistanceRange",  "UserComment",
    "WhiteBalance",  "XResolution",  "YCbCrPositioning",  "YResolution" ];

   var iptc_tags = [ "byline/author", "byline/authorTitle",
    "caption", "captionWriter", "category",
    "city", "contact", "contentLocation/code", "contentLocation/name",
    "copyright", "country", "credit",
    "dateCreated", "digitalCreationDate", "digitalCreationTime",
    "editStatus", "fixtureIdentifier", "headline",
    "instructions", "keyword", "languageId",
    "location", "objectName", "provinceState",
    "source", "subLocation", "supplementalCategory",
    "timeCreated", "transmissionReference", "urgency" ];

   var xmp_tags = [ "RDF/Description" ]
  
  function initForm( mtype, tag, op, search )
  {
    loadTypes( mtype );
    loadTags( tag );
    setButtons( op );
    document.searchForm.search.value = search;
  }

  function loadTypes( mtype )
  {
    var typeList = document.searchForm.mtype;
    var val;
    var opt;
    
    typeList.options.length = 0;
    for( var i=0; i<meta_types.length; i++ )
    {
      val = meta_types[i].split("|");
      opt = new Option( val[0], val[1] );
      if( mtype == val[1] )
        opt.selected = true;
      typeList.options[typeList.options.length] = opt;
    }
  }

  function loadTags( tag )
  {
    var mtype = document.searchForm.mtype.value;
    var tagList = document.searchForm.tag;
    var newTags;
    var opt;
    
    if( mtype == "exif" )
      newTags = exif_tags;
    else if( mtype == "iptc" )
      newTags = iptc_tags;
    else
      newTags = xmp_tags;

    tagList.options.length = 0;

    if( mtype == "exif" || mtype == "iptc" )
    {
      opt = new Option( "--Any tag--", "*" );
      if( tag == "*" )
        opt.selected = true;
      tagList.options[tagList.options.length] = opt;
    }

    for( var i=0; i<newTags.length; i++ )
    {
      opt = new Option( newTags[i], newTags[i] );
      if( tag == newTags[i] )
        opt.selected = true;
      tagList.options[tagList.options.length] = opt;
    }
  }

  function setButtons( op )
  {
    var mtype = document.searchForm.mtype.value;

    if( mtype == "xmp" )
    {
      document.searchForm.op[1].disabled = true;
      document.searchForm.op[1].checked = false;
      document.searchForm.op[0].checked = true;
    } 
    else
    {
      document.searchForm.op[1].disabled = false;
      if( op == "equals" )
        document.searchForm.op[1].checked = true;
      else
        document.searchForm.op[0].checked = true;
    }
  }

  function changeType()
  {
    var mtype = document.searchForm.mtype.value;

    loadTags( mtype, "" );
    setButtons( "" );   
  }
  //  -->
  </script>';

  htp.print( theScript );
END print_script;

---------------------------------------------------------------------------
--  Procedure name:
--      print_search_form
--
--  Description:
--      Output the search form.
---------------------------------------------------------------------------
PROCEDURE print_search_form( mtype  IN VARCHAR2 DEFAULT 'exif',
                             tag    IN VARCHAR2 DEFAULT NULL,
                             op     IN VARCHAR2 DEFAULT 'contains',
                             search IN VARCHAR2 DEFAULT NULL )
IS
  searchForm VARCHAR2(32767);
  f_search   VARCHAR2(64) := substr( search, 1, 40 );
  quotedType VARCHAR2(64);
  quotedTag  VARCHAR2(64);
  selectedTag VARCHAR2(128);
BEGIN

  print_script();

  searchForm := '<p><i>Select the metadata type, the tag, and the search method.
 Enter a word or phrase to search for. Then click the</i> "Search" <i>button.</i>
<p>
  <div class="yellowbg">
  <table summary="Metadata search form" width="100%"
      border=0 cellpadding=4 cellspacing=4>
  <form name="searchForm" method="post" 
        action="PHOTO_ALBUM.SEARCH_METADATA">
  <tr>
   <td scope="col" width="25%" valign="top">
    <label for="mtype_id"><b>Search in metadata:</b></label>
   </td>
   <td scope="col"> 
     <select id="mtype_id" name="mtype" onChange="changeType()">
     </select>
  </tr>
  <tr>
   <td  width="25%" scope="col" valign="top">
    <label for="tag_id"><b>Search in tag:</b></label>
   </td>
   <td scope="col"> 
     <select id="tag_id" name="tag">
     </select>
   </td>
  </tr>
  <tr>
   <td  width="25%" scope="col" valign="top">
    <b>Search method:</b>
   </td>
   <td scope="col">
    <input type="radio" id="contains_id" name="op" value="contains">
     <label for="contains_id">Contains</label></input>' || nbsp || '
    <input type="radio" id="equals_id" name="op" value="equals">
     <label for="equals_id">Equals</lable></input>
   </td>
  </tr>
  <tr>
   <td  width="25%" scope="col" valign="top">
    <label for="search_id"><b>Search string:</b></label>
   </td>
   <td scope="col">
    <input type="text" id="search_id" name="search" maxlength="40">
   </td>
  </tr>
  <tr>
   <td colspan="2"> 
     <input type="submit" value="Search">
   </td>
  </tr>
 </form>
 </table></div>';

 htp.print( searchForm );
 
 searchForm := '
  <script type="text/javascript">
  <!--
   initForm("' || mtype || '", "' || tag ||
           '", "' || op || '", "' || f_search || '")
  // -->
 </script>';
 htp.print( searchForm );

END print_search_form;

---------------------------------------------------------------------------
--  Procedure name:
--      print_upload_form
--
--  Description:
--      Output the upload form.
---------------------------------------------------------------------------
PROCEDURE print_upload_form
IS
  theForm VARCHAR2(4000);
BEGIN
 
  -- the form is...
  theForm := '
<p><i>Enter a description and file location for your photo.
Then click the</i> "Upload photo" <i>button </i>
<p>
<form action="PHOTO_ALBUM.INSERT_NEW_PHOTO" method="post" 
 enctype="multipart/form-data"> 
<div class="yellowbg" >
<table border="0" width="100%" cellspacing="0" summary="photo upload form">
 <tr>
  <td scope="col" colspan="2">' || nbsp || '</td>
 </tr>
 <tr>
  <td scope="col" valign="top"><label for="description_id">
   <b>Description:</b>
   <br><font size="-1">(Optional)</font></label>
  </td>
  <td scope="col" >
   <input id="description_id" type="text" name="new_description" maxlength="40">
   <br><font size="-1">(e.g., My vacation)</font>
  </td>
 </tr>
 <tr>
  <td scope="col" valign="top">
   <label for="filename_id">
    <b>File name:</b>
   </label>
  </td>
  <td scope="col" >
   <input id="filename_id" type="file" name="new_photo">
    <br><font size="-1">(e.g., island.jpg)
  </td>
 </tr>
 <tr>
  <td colspan="2">
   <input type="submit" value="Upload photo" >
  </td>
 </tr>
</table>
</div>
</form></p>';

  htp.print( theForm );

END print_upload_form;

---------------------------------------------------------------------------
--  Procedure name:
--      print_metadata_form
--
--  Description:
--      Output the metadata form.
---------------------------------------------------------------------------
PROCEDURE print_metadata_form( entry_id IN VARCHAR2 )
IS
  theForm VARCHAR2(32667);
BEGIN
  htp.print( '<div>' );
  print_image_link( type=>'thumb', 
                    entry_id=>entry_id, 
                    alt=>'thumbnail of image ' || entry_id,
                    attrs=>'align="left"' );

  htp.print( nbsp || '<i>Complete the form to add metadata to your photo.<br>');
  htp.print( nbsp || 'Then click the</i> "Write it!" <i>button </i>' );
  htp.print( '</div><br clear="left"><p>' );

  --
  -- the form is..
  --
    theForm := '
 <form action="PHOTO_ALBUM.WRITE_METADATA" method="post">
  <input id="entry_id" type="hidden" name="entry_id" value="'|| entry_id || '">
  <div class="yellowbg">
   <table border="0" width="100%" cellspacing="0" summary="metadata input">
    <tr>
     <td scope="col" valign="top">
      <label for="title_id">
       <b>Title:</b>
      </label>
     </td>
     <td scope="col" >
      <input id="title_id" type="text" name="title" maxlength="40">
     </td>
    </tr>

    <tr>
     <td scope="col" valign="top">
      <label for="creator_id">
       <b>Creator:</b>
      <br><span class="smaller">(Optional)</span>
      </label>
     </td>
     <td scope="col" >
      <input id="creator_id" type="text"  name="creator" maxlength="40">
     </td>
    </tr>

    <tr>
     <td scope="col" valign="top">
      <label for="date_id">
      <b>Date:</b>
       <br><span class="smaller">(Optional)</span>
      </label>
     </td>
     <td scope="col" >
      <input id="date_id" type="text" name="date" maxlength="40">
     </td>
    </tr>

    <tr>
     <td scope="col" valign="top">
      <label for="description_id">
       <b>Description:</b>
       <br><span class="smaller">(Optional)</span>
      </label>
     </td>
     <td scope="col" >
      <input id="description_id" type="text" name="description" maxlength="40">
     </td>
    </tr>

    <tr>
     <td scope="col" valign="top">
      <label for="copyright_id">
       <b>Copyright:</b>
       <br><span class="smaller">(Optional)</span>
      </label>
     </td>
     <td scope="col" >
      <input id="copyright_id" type="text" name="copyright" maxlength="40">
     </td>
    </tr>
    <tr><td scope="col" colspan="2">' || nbsp || '</td>
    <tr>
     <td colspan="2" align="left">
      <input type="submit" value="Write it!">
     </td>
    </tr>
  </table>
  </div>
 </form>';

 htp.print( theForm );

END print_metadata_form;

---------------------------------------------------------------------------
--  Procedure name:
--      print_page_header
--
--  Description:
--      Output common header.
---------------------------------------------------------------------------
PROCEDURE print_page_header( heading_tag IN VARCHAR2 DEFAULT NULL )
IS
  style VARCHAR2(4000);
BEGIN
  style := '
 <style type="text/css">
 <!--
  BODY        {background-color:#ffffff}
  .yellowbg   {background-color:#f7f7e7 }
  .blue       {color:#336699 }
  .pageHeader {font-size:x-large; background-color:#f7f7e7; text-align:center }
  .bigBlue    {font-size:large; color:#336699 }
  .navHeader  {font-size:large; font-weight:bold; color:#336699 }
  .smaller    {font-size:small}
 -->
 </style>';

    htp.print( '<html lang="EN">' );
    htp.print( '<head>' );
    htp.title( 'Oracle Multimedia PL/SQL Web Toolkit Photo Album Demo' );

    IF  heading_tag IS NOT NULL 
    THEN
        htp.print( heading_tag );
    END IF;

    htp.print( style );
    htp.print( '</head><body>' );
    htp.print( '<table border="0" width="100%">' );
    htp.print( '<tr><td class="pageHeader">' );
    htp.print( 'Oracle Multimedia PL/SQL Web Toolkit Photo Album Demo');
    htp.print( '</td></tr></table>' );
END print_page_header;

---------------------------------------------------------------------------
--  Procedure name:
--      print_img_link
--
--  Description:
--      Output an <img> tag.
--      Make sure height and width are non-zero when these attributes
--      are included in the <img> tag
---------------------------------------------------------------------------
PROCEDURE print_image_link( type       IN VARCHAR2, 
                            entry_id   IN VARCHAR2,
                            alt        IN VARCHAR2 DEFAULT NULL,
                            height     IN INTEGER  DEFAULT 0, 
                            width      IN INTEGER  DEFAULT 0,
                            attrs      IN VARCHAR2 DEFAULT NULL )
IS
  attributes VARCHAR2(256) := attrs || ' border=1';
  alt2       VARCHAR2(256);
BEGIN
  -- add height and width to tag if non-zero
  IF height > 0 AND width > 0 THEN
    attributes := attributes || ' height=' || height || ' width=' || width;
  END IF;

  -- create an alt text if none given
  IF alt IS NULL THEN
    IF type = 'thumb' THEN
      alt2 := 'thumbnail image ';
    ELSE
      alt2 := 'full-size image ';
    END IF;
    alt2 := alt2 || 'for album entry ' || entry_id;
  ELSE
    alt2 := alt;
  END IF;
  
  htp.img( curl=>'PHOTO_ALBUM.DELIVER_MEDIA?media=' || type ||
                   ampersand || 'entry_id=' || entry_id,
             calt=>alt2, cattributes=>attributes );
END print_image_link;

---------------------------------------------------------------------------
--  Procedure name:
--      print_nav_header
--
--  Description:
--      Output page specific navigation links
---------------------------------------------------------------------------
PROCEDURE print_nav_links( nav_tag  IN VARCHAR2, 
                           entry_id IN VARCHAR2 DEFAULT NULL,
                           search   IN VARCHAR2 DEFAULT NULL )
IS
BEGIN

    htp.print( '<p>' );
    -- create the navigational links 
    CASE

    -- v_ALBUM links to search, upload
    -- display full-text search box
    WHEN 'v_album' = nav_tag THEN
      htp.print( '<span class="navHeader"> View album </span>' );
      htp.print(  nbsp || '[' || nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_SEARCH_FORM', 'Search metadata' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_UPLOAD_FORM', 'Upload photo'  );
      htp.prn( nbsp || ']<p>' );
      htp.formOpen( 'PHOTO_ALBUM.VIEW_ALBUM', 'GET' );
      htp.print( '<label>Full text search:' || nbsp );
      htp.formText( 'q', 40, 40, '' );
      htp.print( '</label>' );
      htp.formSubmit( cvalue=>'Search' );
      htp.formClose;

    -- S_ALBUM links to view album, search, upload
    -- display full-text search box
    WHEN 's_album' = nav_tag THEN
      htp.print( '<span class="navHeader">Search album</span>' );
      htp.print(  nbsp || '[' || nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_ALBUM', 'View album' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_SEARCH_FORM', 'Search metadata' );
      htp.print( nbsp );
      htp.anchor( curl=>'PHOTO_ALBUM.VIEW_UPLOAD_FORM', 
                  ctext=>'Upload photo'  );
      htp.prn( nbsp || ']<p>' );
      htp.formOpen( 'PHOTO_ALBUM.VIEW_ALBUM', 'GET' );
      htp.print( '<label>Full text search:' || nbsp );
      htp.formText( 'q', 40, 40, nvl(search, '') );
      htp.print( '</label>' );
      htp.formSubmit( cvalue=>'Search' );
      htp.formClose;

    -- ENTRY links to view metadata, write metadata, album, search, upload
    WHEN 'entry' = nav_tag THEN
      htp.print( '<span class="navHeader">View entry</span>' );
      htp.print(  nbsp || '[' || nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_METADATA?entry_id='|| entry_id, 
                  'View metadata' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_METADATA_FORM?entry_id='|| entry_id, 
                  'Write metadata' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_ALBUM', 'View album' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_SEARCH_FORM', 'Search metadata' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_UPLOAD_FORM', 'Upload photo'  );
      htp.print( nbsp || ']' );

    -- VIEW METADATA links to entry, write metadata, album, search, upload
    WHEN 'view_metadata' = nav_tag THEN
      htp.print( '<span class="navHeader">View metadata</span>' );
      htp.print(  nbsp || '[' || nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_ENTRY?entry_id=' || entry_id, 
                  'View entry' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_METADATA_FORM?entry_id='|| entry_id, 
                  'Write metadata' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_ALBUM', 'View album' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_SEARCH_FORM', 'Search metadata' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_UPLOAD_FORM', 'Upload photo'  );
      htp.print( nbsp || ']' );

    -- WRITE METADATA links to entry, view metadata album, search, upload
    WHEN 'write_metadata' = nav_tag THEN
      htp.print( '<span class="navHeader">Write XMP metadata</span>' );
      htp.print(  nbsp || '[' || nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_ENTRY?entry_id=' || entry_id, 
                  'View entry' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_METADATA?entry_id='|| entry_id, 
                  'View metadata' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_ALBUM', 'View album' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_SEARCH_FORM', 'Search metadata' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_UPLOAD_FORM', 'Upload photo'  );
      htp.print( nbsp || ']' );

    -- SEARCH METADATA links to album, upload
    WHEN 'search_metadata' = nav_tag THEN
      htp.print( '<span class="navHeader">Search metadata</span>' );
      htp.print(  nbsp || '[' || nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_ALBUM', 'View album' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_UPLOAD_FORM', 'Upload photo'  );
      htp.print( nbsp || ']' );

    -- UPLOAD links to album, search
    WHEN 'upload' = nav_tag THEN
      htp.print( '<span class="navHeader">Upload photo</span>' );
      htp.print(  nbsp || '[' || nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_ALBUM', 'View album' );
      htp.print( nbsp );
      htp.anchor( 'PHOTO_ALBUM.VIEW_SEARCH_FORM', 'Search metadata' );
      htp.print( nbsp || ']' );

    ELSE
      NULL;
    END CASE;
    htp.hr( cattributes=>'size="-1"' );

END print_nav_links;


---------------------------------------------------------------------------
--  Procedure name:
--      print_page_trailer
--
--  Description:
--      Output common trailer.
---------------------------------------------------------------------------
PROCEDURE print_page_trailer( display_return_url IN BOOLEAN DEFAULT FALSE ) IS
BEGIN
    IF display_return_url
    THEN
        print_link( 'PHOTO_ALBUM.VIEW_ALBUM', 'Return to photo album' );
    END IF;
    htp.print( '</body></html>' );
END print_page_trailer;


---------------------------------------------------------------------------
--  Procedure name:
--      print_heading
--
--  Description:
--      Output a one-line heading.
---------------------------------------------------------------------------
PROCEDURE print_heading( message IN VARCHAR2 ) IS
BEGIN
    htp.print( '<p><span class="blue">' );
    htp.print( '<b>' || message || '</b></span></p>' );
END print_heading;


---------------------------------------------------------------------------
--  Procedure name:
--      print_error
--
--  Description:
--      Output an error message
---------------------------------------------------------------------------
PROCEDURE print_error( message IN VARCHAR2 ) IS
BEGIN
    htp.print( '<p><span class="bigBlue">' );
    htp.print( '<b>Error:</b></span>' || nbsp || message );
END print_error;


---------------------------------------------------------------------------
--  Procedure name:
--      print_link
--
--  Description:
--      Output an anchor tag.
---------------------------------------------------------------------------
PROCEDURE print_link( link IN VARCHAR2, message IN VARCHAR2 ) IS
BEGIN
    htp.print( '<p><table class="yellowbg" width="100%">' );
    htp.print( '<tr><td colspan="3" align="center">' );
    htp.print( '<a href="' || link || '">' || message || '</a>' ||
               '</td></tr></table></p>' );
END print_link;
 
---------------------------------------------------------------------------
--  Procedure name:
--      print_metadata
--
--  Description:
--      Print image metadata in a <pre> tag
---------------------------------------------------------------------------
PROCEDURE print_metadata( meta in XMLType ) IS
    metaClob  CLOB    := null;
    len       INTEGER := 0;
    buf       VARCHAR2(8192);
    bufSize   INTEGER := 8192;
    pos       INTEGER := 1;
BEGIN
    metaClob := getClob(meta);
  
    IF metaClob IS NULL THEN
      return;
    END IF;

    len := dbms_lob.getLength( metaClob );
    IF bufSize > len THEN
      bufSize := len;
    END IF;
    WHILE len > 0 LOOP
      dbms_lob.read( metaClob, bufSize, pos, buf );
      htp.prints( buf );
      pos := pos + bufSize;
      len := len - bufSize;
    END LOOP;

    IF( dbms_lob.isTemporary(metaClob) != 0 )  THEN
      dbms_lob.freeTemporary(metaCLob);
    END IF;

END print_metadata;


---------------------------------------------------------------------------
--  Function name:
--      getClob
--
--  Description:
--      returns the CLOB of an XMLType.
--      used to create a multi-column datastore for full-text indexing
---------------------------------------------------------------------------
FUNCTION getClob( xmlDoc IN XMLType ) RETURN CLOB IS
  c clob;
BEGIN
  IF xmlDoc IS NOT NULL THEN
    select xmlSerialize(document xmldoc as clob indent size=2) into c 
    from dual;
    RETURN c;
  ELSE
    RETURN NULL;
  END IF;
END getClob;

-- Specify this function for the PlsqlRequestValidationFunction
-- parameter in mod_plsql DAD configuration. Restricts the PhotoAlbum DAD
-- to execute only the photo_album package.
FUNCTION validate_request( procedure_name IN VARCHAR2 ) RETURN BOOLEAN IS
BEGIN
  IF( upper(procedure_name) like 'PHOTO_ALBUM.%' ) THEN
    return TRUE;
  ELSE
    return FALSE;
  END IF;
END validate_request;

END photo_album;
/
SHOW ERRORS;

